<?php
/**
 * Class Aigentic_Admin_Controller
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Aigentic_Admin_Controller {

    public function __construct() {
        add_action( 'admin_menu', [ $this, 'register_aigentic_menu' ] );
        add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_admin_assets' ] );
        
        add_filter( 'woocommerce_product_data_tabs', [ $this, 'add_aigentic_product_tab' ] );
        add_action( 'woocommerce_product_data_panels', [ $this, 'render_aigentic_tab_content' ] );
        add_action( 'woocommerce_process_product_meta', [ $this, 'save_ai_context_field' ] );

        add_action( 'wp_ajax_aigentic_verify_gemini', [ $this, 'ajax_test_gemini_connection' ] );
        add_action( 'wp_ajax_aigentic_generate_summary', [ $this, 'ajax_generate_ai_summary' ] );
        add_action( 'wp_ajax_aigentic_save_language', [ $this, 'ajax_save_language' ] );
    }

    public function register_aigentic_menu() {
        add_menu_page( 
            esc_html__( 'Aigentic AI', 'aigentic' ), 
            esc_html__( 'Aigentic AI', 'aigentic' ), 
            'manage_options', 
            'aigentic', 
            [ $this, 'render_dashboard' ], 
            'dashicons-rest-api', 
            58 
        );
    }

    public function enqueue_admin_assets( $hook ) {
        if ( 'toplevel_page_aigentic' !== $hook && 'post.php' !== $hook && 'post-new.php' !== $hook ) return;

        // We load the CSS, but we will handle critical JS inline to avoid caching issues
        wp_enqueue_style( 'aigentic-admin-css', AIGENTIC_URL . 'assets/css/admin.css', [], AIGENTIC_VERSION );
        
        // Load jQuery for our inline script
        wp_enqueue_script( 'jquery' );
    }

    public function render_dashboard() {
        global $wpdb;
        $table = $wpdb->prefix . 'aigentic_bot_logs';
        
        // FIX: Implemented caching (Transient) to solve "DirectDatabaseQuery.NoCaching" warning
        $stats = get_transient( 'aigentic_dashboard_stats' );
        
        if ( false === $stats ) {
            $stats = [];
            
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $table_exists = $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $table ) );
            
            if ( $table_exists === $table ) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
                $results = $wpdb->get_results( "SELECT bot_name, COUNT(*) as count FROM {$table} GROUP BY bot_name" );
                if ( $results ) {
                    $stats = $results;
                }
            }
            // Cache results for 1 hour
            set_transient( 'aigentic_dashboard_stats', $stats, HOUR_IN_SECONDS );
        }
        
        $active_model = get_option( 'aigentic_gemini_model', 'Not Set' );
        $saved_lang   = get_option( 'aigentic_target_language', 'English' );
        $languages    = [ 'English', 'Spanish', 'French', 'German', 'Greek', 'Italian', 'Portuguese', 'Dutch', 'Japanese', 'Chinese', 'Korean' ];
        ?>
        <div class="wrap" style="max-width: 1100px;">
            <div style="background: #7c3aed; padding: 40px; border-radius: 12px; color: white; margin-bottom: 30px;">
                <h1 style="color: white; margin: 0; font-size: 32px;"><?php esc_html_e( 'Aigentic AI Foundation', 'aigentic' ); ?></h1>
                <p style="opacity: 0.9; margin: 10px 0 0;"><?php esc_html_e( 'Commerce-to-Agent Foundation', 'aigentic' ); ?></p>
            </div>

            <div style="display: grid; grid-template-columns: 2fr 1fr; gap: 30px;">
                <div>
                    <div style="background: white; padding: 30px; border-radius: 12px; border: 1px solid #e2e8f0; margin-bottom: 30px;">
                        <h2 style="margin-top: 0;"><?php esc_html_e( 'Gemini Configuration', 'aigentic' ); ?></h2>
                        
                        <label style="font-weight: bold; display: block; margin-bottom: 5px;"><?php esc_html_e( 'Google Gemini API Key', 'aigentic' ); ?></label>
                        <div style="display: flex; gap: 10px; margin-bottom: 20px;">
                            <input type="password" id="aigentic_gemini_key" value="<?php echo esc_attr( get_option( 'aigentic_gemini_key' ) ); ?>" placeholder="<?php esc_attr_e( 'Paste API Key Here', 'aigentic' ); ?>" style="flex-grow: 1; padding: 10px; border-radius: 6px; border: 1px solid #cbd5e1;">
                            <button id="aigentic-test-api" class="button button-primary" style="background: #7c3aed; border: none; height: 42px; border-radius: 6px;">
                                <?php esc_html_e( 'Save & Test', 'aigentic' ); ?>
                            </button>
                        </div>

                        <label style="font-weight: bold; display: block; margin-bottom: 5px;"><?php esc_html_e( 'Content Generation Language', 'aigentic' ); ?></label>
                        <div style="display: flex; gap: 10px; margin-bottom: 10px;">
                            <select id="aigentic_target_language" style="width: 100%; max-width: 300px; padding: 8px; border-radius: 6px; border: 1px solid #cbd5e1;">
                                <?php foreach ( $languages as $lang ) : ?>
                                    <option value="<?php echo esc_attr( $lang ); ?>" <?php selected( $saved_lang, $lang ); ?>>
                                        <?php echo esc_html( $lang ); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                            <button id="aigentic-save-lang" class="button" style="height: 38px;"><?php esc_html_e( 'Save Language', 'aigentic' ); ?></button>
                        </div>
                        <p style="font-size: 13px; color: #64748b; margin-top: 0;"><?php esc_html_e( 'This language will be used for all AI descriptions (Vision & Bulk).', 'aigentic' ); ?></p>
                        
                        <hr style="margin: 20px 0; border: 0; border-top: 1px solid #e2e8f0;">

                        <p><strong><?php esc_html_e( 'Active Model:', 'aigentic' ); ?></strong> <code><?php echo esc_html( $active_model ); ?></code></p>
                        <div id="aigentic-api-feedback" style="margin-top: 15px; font-weight: bold;"></div>
                    </div>

                    <?php do_action( 'aigentic_dashboard_after_api' ); ?>

                    <div style="background: white; padding: 30px; border-radius: 12px; border: 1px solid #e2e8f0;">
                        <h2 style="margin-top: 0;"><?php esc_html_e( 'AI Agent Statistics', 'aigentic' ); ?></h2>
                        <?php if ( ! empty( $stats ) ) : ?>
                            <table class="widefat striped" style="border:none;">
                                <thead><tr><th><?php esc_html_e( 'Bot Name', 'aigentic' ); ?></th><th><?php esc_html_e( 'Visits', 'aigentic' ); ?></th></tr></thead>
                                <tbody>
                                    <?php foreach ( $stats as $s ) : ?>
                                        <tr>
                                            <td><strong><?php echo esc_html( $s->bot_name ); ?></strong></td>
                                            <td><?php echo esc_html( $s->count ); ?></td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        <?php else : ?>
                            <p style="color: #64748b;"><?php esc_html_e( 'Waiting for AI bots to discover your store...', 'aigentic' ); ?></p>
                        <?php endif; ?>
                    </div>
                </div>

                <div>
                    <div style="background: #1e293b; padding: 30px; border-radius: 12px; color: white;">
                        <h2 style="margin-top: 0; color: white;"><?php esc_html_e( 'Unlock Pro', 'aigentic' ); ?></h2>
                        <ul style="padding: 0; list-style: none; margin: 20px 0; font-size: 14px; line-height: 1.6;">
                            <li style="margin-bottom: 10px;">✅ <?php esc_html_e( 'Gemini Vision AI (Image Analysis)', 'aigentic' ); ?></li>
                            <li style="margin-bottom: 10px;">✅ <?php esc_html_e( 'Automated Bulk Rewrites', 'aigentic' ); ?></li>
                            <li style="margin-bottom: 10px;">✅ <?php esc_html_e( 'Full Analytics History', 'aigentic' ); ?></li>
                        </ul>
                        <a href="https://ikaros.pro" target="_blank" style="display: block; text-align: center; background: #7c3aed; color: white; text-decoration: none; padding: 12px; border-radius: 8px; font-weight: 700;">
                            <?php esc_html_e( 'Upgrade to Pro', 'aigentic' ); ?> &rarr;
                        </a>
                    </div>
                </div>
            </div>
        </div>

        <script type="text/javascript">
        jQuery(document).ready(function($) {
            
            // 1. SAVE LANGUAGE HANDLER
            $('#aigentic-save-lang').on('click', function(e) {
                e.preventDefault();
                var btn = $(this);
                var lang = $('#aigentic_target_language').val();
                
                btn.text('Saving...').prop('disabled', true);

                $.post(ajaxurl, {
                    action: 'aigentic_save_language',
                    language: lang,
                    // FIX: Escaped output with esc_js()
                    nonce: '<?php echo esc_js( wp_create_nonce( 'aigentic_admin_nonce' ) ); ?>'
                }, function(response) {
                    if (response.success) {
                        btn.text('Saved!');
                        setTimeout(function(){ btn.text('Save Language').prop('disabled', false); }, 2000);
                    } else {
                        btn.text('Error').prop('disabled', false);
                        alert('Error saving language');
                    }
                });
            });

            // 2. TEST API HANDLER
            $('#aigentic-test-api').on('click', function(e) {
                e.preventDefault();
                var btn = $(this);
                var key = $('#aigentic_gemini_key').val();
                var feedback = $('#aigentic-api-feedback');

                if(!key) { alert('Please enter an API Key'); return; }

                btn.text('Connecting...').prop('disabled', true);
                feedback.html('');

                $.post(ajaxurl, {
                    action: 'aigentic_verify_gemini',
                    api_key: key,
                    // FIX: Escaped output with esc_js()
                    nonce: '<?php echo esc_js( wp_create_nonce( 'aigentic_admin_nonce' ) ); ?>'
                }, function(response) {
                    btn.text('Save & Test').prop('disabled', false);
                    if (response.success) {
                        feedback.html('<span style="color:#047857;">✅ ' + response.data + '</span>');
                    } else {
                        feedback.html('<span style="color:#b91c1c;">❌ ' + (response.data || 'Connection failed') + '</span>');
                    }
                });
            });

        });
        </script>
        <?php
    }

    public function add_aigentic_product_tab( $tabs ) {
        $tabs['aigentic_ai'] = [
            'label'    => __( 'Aigentic AI', 'aigentic' ),
            'target'   => 'aigentic_ai_data_panel',
            'class'    => [ 'show_if_simple', 'show_if_variable' ],
            'priority' => 99,
        ];
        return $tabs;
    }

    public function render_aigentic_tab_content() {
        global $post;
        $summary = get_post_meta( $post->ID, '_aigentic_ai_summary', true );

        echo '<div id="aigentic_ai_data_panel" class="panel woocommerce_options_panel hidden">';
        echo '<div class="options_group">';
        
        wp_nonce_field( 'aigentic_save_product_data', 'aigentic_product_nonce' );

        echo '<h3 style="padding: 10px 10px 0;">' . esc_html__( 'AI Agent Optimization', 'aigentic' ) . '</h3>';
        
        woocommerce_wp_textarea_input([
            'id'          => '_aigentic_ai_summary',
            'label'       => __( 'AI Context Summary', 'aigentic' ),
            'value'       => $summary, 
            'placeholder' => __( 'Click generate to analyze product...', 'aigentic' ),
            'description' => __( 'This summary is optimized for AI agents to understand your product value.', 'aigentic' ),
            'desc_tip'    => true,
            'style'       => 'width: 70%; height: 100px;'
        ]);

        echo '<p class="form-field">
                <button type="button" id="aigentic-gen-btn" class="button button-primary" style="margin-left: 161px;">
                    ' . esc_html__( 'Generate with Gemini AI', 'aigentic' ) . '
                </button>
              </p>';

        echo '<div style="margin: 20px 20px 20px 161px; background: #fffbeb; padding: 15px; border: 1px solid #fde68a; border-radius: 8px; max-width: 600px;">';
        echo '<p style="margin: 0; color: #92400e;"><strong>' . esc_html__( 'Want smarter AI?', 'aigentic' ) . '</strong> ' . esc_html__( 'This generator uses text-only analysis. Upgrade to Aigentic Pro to unlock Vision AI.', 'aigentic' ) . '</p>';
        echo '<a href="https://ikaros.pro" target="_blank" style="display: inline-block; margin-top: 10px; color: #b45309; text-decoration: none; font-weight: 700;">';
        echo esc_html__( '🚀 Upgrade to Pro for Vision AI Metadata', 'aigentic' ) . ' &rarr;';
        echo '</a>';
        echo '</div>';

        echo '</div>';
        echo '</div>';
        
        // Inline JS for Product Page Button
        ?>
        <script type="text/javascript">
        jQuery(document).ready(function($) {
            $('#aigentic-gen-btn').on('click', function(e) {
                e.preventDefault();
                var btn = $(this);
                btn.text('Analyzing...').prop('disabled', true);
                
                $.post(ajaxurl, {
                    action: 'aigentic_generate_summary',
                    // FIX: Escaped ID with absint()
                    product_id: <?php echo absint( $post->ID ); ?>,
                    // FIX: Escaped output with esc_js()
                    nonce: '<?php echo esc_js( wp_create_nonce( 'aigentic_admin_nonce' ) ); ?>'
                }, function(response) {
                    btn.text('Generate with Gemini AI').prop('disabled', false);
                    if (response.success && response.data.summary) {
                        $('#_aigentic_ai_summary').val(response.data.summary);
                    } else {
                        alert('Error: ' + (response.data || 'Unknown error'));
                    }
                });
            });
        });
        </script>
        <?php
    }

    public function save_ai_context_field( $post_id ) {
        if ( ! isset( $_POST['aigentic_product_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['aigentic_product_nonce'] ) ), 'aigentic_save_product_data' ) ) {
            return;
        }

        if ( ! current_user_can( 'edit_post', $post_id ) ) {
            return;
        }

        if ( isset( $_POST['_aigentic_ai_summary'] ) ) {
            update_post_meta( $post_id, '_aigentic_ai_summary', sanitize_textarea_field( wp_unslash( $_POST['_aigentic_ai_summary'] ) ) );
        }
    }

    public function ajax_save_language() {
        check_ajax_referer( 'aigentic_admin_nonce', 'nonce' );
        
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Unauthorized' );
        }

        if ( isset( $_POST['language'] ) ) {
            $lang = sanitize_text_field( wp_unslash( $_POST['language'] ) );
            update_option( 'aigentic_target_language', $lang );
            wp_send_json_success();
        } else {
            wp_send_json_error( 'No language provided' );
        }
    }

    public function ajax_test_gemini_connection() {
        check_ajax_referer( 'aigentic_admin_nonce', 'nonce' );
        
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Unauthorized' );
        }

        $key = isset( $_POST['api_key'] ) ? sanitize_text_field( wp_unslash( $_POST['api_key'] ) ) : '';
        
        if ( empty( $key ) ) wp_send_json_error( 'Key is missing.' );

        $list_url = "https://generativelanguage.googleapis.com/v1beta/models?key=" . $key;
        $response = wp_remote_get( $list_url );
        
        if ( is_wp_error( $response ) ) {
            wp_send_json_error( 'Connection failed' );
        }

        $body = json_decode( wp_remote_retrieve_body( $response ), true );
        
        $best_model = 'gemini-1.5-flash'; 
        if ( isset( $body['models'] ) ) {
            foreach ( $body['models'] as $model ) {
                if ( isset($model['supportedGenerationMethods']) && in_array('generateContent', $model['supportedGenerationMethods']) ) {
                    $name = sanitize_text_field( str_replace('models/', '', $model['name']) );
                    if ( strpos($name, 'flash') !== false ) {
                        $best_model = $name;
                        break;
                    }
                }
            }
        }

        $test_url = "https://generativelanguage.googleapis.com/v1beta/models/{$best_model}:generateContent?key=" . $key;
        $test_res = wp_remote_post( $test_url, [
            'headers' => ['Content-Type' => 'application/json'],
            'body'    => json_encode([ 'contents' => [[ 'parts' => [[ 'text' => 'Hello' ]] ]] ])
        ]);

        if ( is_wp_error( $test_res ) || wp_remote_retrieve_response_code( $test_res ) !== 200 ) {
            wp_send_json_error( "Handshake Failed with $best_model" );
        }

        update_option( 'aigentic_gemini_key', $key );
        update_option( 'aigentic_gemini_model', $best_model );
        
        wp_send_json_success( "Connected to: " . esc_html( $best_model ) );
    }

    public function ajax_generate_ai_summary() {
        check_ajax_referer( 'aigentic_admin_nonce', 'nonce' );
        
        if ( ! current_user_can( 'edit_products' ) ) {
            wp_send_json_error( 'Unauthorized' );
        }

        if ( ob_get_length() ) ob_clean();

        $id = isset( $_POST['product_id'] ) ? intval( wp_unslash( $_POST['product_id'] ) ) : 0;
        
        $key = get_option( 'aigentic_gemini_key' );
        $model = get_option( 'aigentic_gemini_model', 'gemini-1.5-flash' );

        if ( ! $key ) wp_send_json_error( 'No API Key found.' );

        $product = wc_get_product( $id );
        if ( ! $product ) wp_send_json_error( 'Product not found' );
        
        $language = get_option( 'aigentic_target_language', 'English' );

        $desc = wp_strip_all_tags( $product->get_short_description() );
        if ( empty( $desc ) ) $desc = wp_strip_all_tags( $product->get_description() );

        $prompt = "Summarize this product in 2 sentences in **" . $language . "** for an AI shopping agent: " . $product->get_name() . ". Info: " . $desc;

        $url = "https://generativelanguage.googleapis.com/v1beta/models/{$model}:generateContent?key=" . $key;
        
        $response = wp_remote_post( $url, [
            'headers' => ['Content-Type' => 'application/json'],
            'body'    => json_encode([ 'contents' => [[ 'parts' => [[ 'text' => $prompt ]] ]] ]),
            'timeout' => 30
        ]);

        if ( is_wp_error( $response ) ) wp_send_json_error( 'API Error' );

        $body = json_decode( wp_remote_retrieve_body( $response ), true );
        $text = $body['candidates'][0]['content']['parts'][0]['text'] ?? '';

        if ( ob_get_length() ) ob_clean();

        if ( $text ) {
            wp_send_json_success([ 'summary' => wp_kses_post( $text ) ]);
        } else {
            wp_send_json_error( 'AI returned empty response.' );
        }
    }
    
}
   add_action( 'wp_head', 'aigentic_inject_ai_schema' );
function aigentic_inject_ai_schema() {
    if ( is_product() ) {
        global $post;
        $product = wc_get_product( $post->ID );
        
        // Retrieve our AI context and product details
        $ai_summary = get_post_meta( $post->ID, '_aigentic_ai_summary', true );
        $image_url  = wp_get_attachment_url( $product->get_image_id() );

        $schema = [
            "@context" => "https://schema.org/",
            "@type"    => "Product",
            "name"     => $product->get_name(),
            "image"    => $image_url,
            "description" => $ai_summary ?: $product->get_short_description(),
            "sku"      => $product->get_sku() ?: 'AIG-' . $post->ID,
            "brand"    => [
                "@type" => "Brand",
                "name"  => get_bloginfo( 'name' )
            ],
            "offers"   => [
                "@type"         => "Offer",
                "url"           => get_permalink( $post->ID ),
                "priceCurrency" => get_woocommerce_currency(),
                "price"         => $product->get_price(),
                "availability"  => $product->is_in_stock() ? "https://schema.org/InStock" : "https://schema.org/OutOfStock",
                "itemCondition" => "https://schema.org/NewCondition"
            ]
        ];

        echo '<script type="application/ld+json">' . json_encode( $schema ) . '</script>';
    }
}