<?php
/**
 * Class Aigentic_Bot_Bridge
 * Handles Discovery (Robots.txt), Secure Routing, AND Bot Statistics.
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Aigentic_Bot_Bridge {

    public function __construct() {
        add_filter( 'robots_txt', [ $this, 'inject_ai_rules' ], 20, 2 );
        add_action( 'admin_notices', [ $this, 'detect_physical_robots_conflict' ] );
        add_action( 'send_headers', [ $this, 'send_discovery_headers' ] );
        add_action( 'init', [ $this, 'handle_agentic_purchase_request' ] );
        add_action( 'wp', [ $this, 'track_bot_visit' ] );
    }

    public function inject_ai_rules( $output, $public ) {
        $rules  = "\n# --- Aigentic AI Permission Rules ---\n";
        $rules .= "User-agent: GPTBot\nAllow: /.well-known/ai-commerce.json\n";
        $rules .= "User-agent: Google-Extended\nAllow: /.well-known/ai-commerce.json\n";
        $rules .= "User-agent: PerplexityBot\nAllow: /.well-known/ai-commerce.json\n";
        $rules .= "User-agent: *\nAllow: /.well-known/ai-commerce.json\n";
        return $output . apply_filters( 'aigentic_robots_filter', $rules );
    }

    public function detect_physical_robots_conflict() {
        if ( ! current_user_can( 'manage_options' ) ) return;
        
        $physical_file = ABSPATH . 'robots.txt';
        if ( file_exists( $physical_file ) ) {
            /* translators: %s: path to robots.txt */
            $message = sprintf( __( 'Aigentic Alert: A physical %s file was detected. Please add "Allow: /.well-known/ai-commerce.json" manually.', 'aigentic' ), '<code>robots.txt</code>' );
            echo '<div class="notice notice-warning is-dismissible"><p>' . wp_kses_post( $message ) . '</p></div>';
        }
    }

    public function send_discovery_headers() {
        if ( is_admin() ) return;
        
        $manifest_url = home_url( '/.well-known/ai-commerce.json' );
        header( 'X-Aigentic-Manifest: ' . esc_url_raw( $manifest_url ) );
        header( 'Access-Control-Allow-Origin: *' ); 
        
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        if ( isset( $_GET['aigentic_manifest'] ) || is_shop() || is_product() ) {
            header( 'X-Robots-Tag: index, follow, all' );
        }
    }

    public function handle_agentic_purchase_request() {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        if ( ! isset( $_GET['aigentic_buy'] ) ) return;

        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $product_id = absint( $_GET['aigentic_buy'] );
        $product    = wc_get_product( $product_id );

        if ( ! $product || ! $product->is_purchasable() ) return;

        setcookie( 'aigentic_bot_ref', '1', time() + ( DAY_IN_SECONDS * 30 ), COOKIEPATH, COOKIE_DOMAIN, is_ssl(), true );

        if ( function_exists( 'WC' ) && isset( WC()->cart ) ) {
            WC()->cart->empty_cart();
            WC()->cart->add_to_cart( $product_id );
            $checkout_url = add_query_arg( [
                'utm_source'   => 'aigentic',
                'utm_medium'   => 'ai_agent',
                'utm_campaign' => 'agentic_purchase'
            ], wc_get_checkout_url() );
            wp_safe_redirect( $checkout_url );
            exit;
        }
    }

    public function track_bot_visit() {
        if ( is_admin() ) return;

        $user_agent  = isset( $_SERVER['HTTP_USER_AGENT'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ) ) : '';
        $request_uri = isset( $_SERVER['REQUEST_URI'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : '/';
        $bot_name    = '';

        if ( stripos( $user_agent, 'Googlebot' ) !== false ) {
            $bot_name = 'Google Search';
        } elseif ( stripos( $user_agent, 'Google-Extended' ) !== false ) {
            $bot_name = 'Google Gemini';
        } elseif ( stripos( $user_agent, 'GPTBot' ) !== false ) {
            $bot_name = 'ChatGPT';
        } elseif ( stripos( $user_agent, 'ChatGPT-User' ) !== false ) {
            $bot_name = 'ChatGPT (Search)';
        } elseif ( stripos( $user_agent, 'ClaudeBot' ) !== false ) {
            $bot_name = 'Claude AI';
        } elseif ( stripos( $user_agent, 'Perplexity' ) !== false ) {
            $bot_name = 'Perplexity';
        } elseif ( stripos( $user_agent, 'bingbot' ) !== false ) {
            $bot_name = 'Bing AI';
        } elseif ( isset( $_GET['test_bot_stats'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
            $bot_name = 'Test Bot';
        }

        if ( ! empty( $bot_name ) ) {
            global $wpdb;
            $table = $wpdb->prefix . 'aigentic_bot_logs';
            
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery
            $table_exists = $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $table ) );

            if ( $table_exists === $table ) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery
                $wpdb->insert( $table, [ 
                    'bot_name'          => $bot_name, 
                    'user_agent'        => substr( $user_agent, 0, 255 ),
                    'endpoint_accessed' => $request_uri,
                    'visit_time'        => current_time( 'mysql' )
                ]);
            }
        }
    }
}