<?php
/**
 * Class Aigentic_Manifest_Server
 * Generates the clean, real-time JSON catalog for AI Agent consumption.
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Aigentic_Manifest_Server {

    public function __construct() {
        // 1. Setup the Endpoint
        add_action( 'init', [ $this, 'register_manifest_rewrite' ] );
        add_filter( 'query_vars', [ $this, 'add_manifest_query_var' ] );

        // 2. High-Priority Delivery
        add_action( 'template_redirect', [ $this, 'dispatch_manifest' ], 1 );
    }

    public function register_manifest_rewrite() {
        add_rewrite_rule( '^.well-known/ai-commerce.json$', 'index.php?aigentic_manifest=1', 'top' );
    }

    public function add_manifest_query_var( $vars ) {
        $vars[] = 'aigentic_manifest';
        return $vars;
    }

    public function dispatch_manifest() {
        if ( get_query_var( 'aigentic_manifest' ) !== '1' ) {
            return;
        }

        // 1. Clear Buffer
        if ( ob_get_length() ) {
            ob_clean();
        }

        // 2. LOGGING: Use the central Bot Bridge if available
        // This ensures the Dashboard Stats are accurate.
        if ( class_exists( 'Aigentic_Bot_Bridge' ) ) {
    $bridge = new Aigentic_Bot_Bridge();
    $bridge->track_bot_visit(); // Correct function name
}

        // 3. Headers
        header( 'Content-Type: application/json; charset=utf-8' );
        header( 'Access-Control-Allow-Origin: *' );
        header( 'X-Robots-Tag: noindex, follow' );
        header( 'Cache-Control: no-store, no-cache, must-revalidate, max-age=0' );

        // 4. Data Compilation
        $catalog = [];
        $products = wc_get_products( [
            'status' => 'publish',
            'limit'  => 100, 
            'orderby' => 'date',
            'order'   => 'DESC',
        ] );

        foreach ( $products as $product ) {
            // Allow Pro plugin to override this
            $ai_description = apply_filters( 'aigentic_product_summary', $product->get_short_description(), $product->get_id() );
            
            // Fallback to post meta if filter didn't run
            if ( empty( $ai_description ) ) {
                 $ai_description = get_post_meta( $product->get_id(), '_aigentic_ai_summary', true );
            }
            // Final fallback
            if ( empty( $ai_description ) ) {
                $ai_description = wp_trim_words( $product->get_description(), 35 );
            }

            $catalog[] = [
                'name'         => html_entity_decode( $product->get_name() ),
                'sku'          => $product->get_sku(),
                'price'        => (float) $product->get_price(),
                'currency'     => get_woocommerce_currency(),
                'url'          => get_permalink( $product->get_id() ),
                'purchase_url' => add_query_arg( 'aigentic_buy', $product->get_id(), home_url( '/' ) ),
                'image'        => wp_get_attachment_url( $product->get_image_id() ),
                'description'  => esc_html( $ai_description ),
                'availability' => $product->is_in_stock() ? 'in_stock' : 'out_of_stock',
            ];
        }

        // 5. Output
        echo json_encode( [
            'manifest_version' => AIGENTIC_VERSION,
            'plugin_type'      => 'Free Foundation',
            'store_info'       => [
                'name' => get_bloginfo( 'name' ),
                'url'  => home_url(),
            ],
            'catalog'          => $catalog,
            'last_updated'     => current_time( 'mysql' ),
        ], JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES );

        exit;
    }
}